CL_DEBUG = false -- for internal testing only, leave it set to false!

local playerGUID, hasSpecialAura, inVehicle

--[[ global addon variables ]]
SUBHEADER_TEXT_COLOR  = "|cffCEA208"
CRITLINE_DMG_ICON = "Interface\\Icons\\Ability_SteelMelee"
CRITLINE_HEAL_ICON = "Interface\\Icons\\Spell_Holy_FlashHeal"
CRITLINE_PET_ICON = "Interface\\Icons\\Ability_Hunter_Pet_Bear"

CL_CRIT_TEXT = strmatch(TEXT_MODE_A_STRING_RESULT_CRITICAL, "(%a+)")

Critline_Loaded = false
Critline_UpdateTimer = 0
Critline_OutOfControl = false
Critline_MindControlled = false


function Critline_DataReset()
	CL_DATABASE = {}
	Critline_Summary = {DMG = "", HEAL = "", PET = ""}
	Critline_Initialize()
end


function Critline_SettingsReset()
	CL_SETTINGS = {}
	Critline_Initialize()
end


function Critline_ResetAll()
	CL_SETTINGS = {}
	CL_DATABASE = {}
	Critline_Summary = {DMG = "", HEAL = "", PET = ""}
	Critline_Initialize()
end


function Critline_GetHighest(tree, hittype)
	-- hittype is 'NORM' or 'CRIT'
	-- tree is 'DMG', 'PET' or 'HEAL'
	
	local hidmg = 0
	local spell = "n/a"
	
	tree = strupper(tree)
	if not CL_DATABASE then 
		return hidmg, spell
	end

	if not CL_DATABASE[tree] then 
		return hidmg, spell
	end

	for k, v in pairs (CL_DATABASE[tree]) do
		if CL_DATABASE[tree][k][hittype] then
			if CL_DATABASE[tree][k][hittype]["Amount"] > hidmg then
				if (CL_SETTINGS["INVERT"..tree] == 0 and not CritlineFilters_IsSpellInFilter(k, tree)) or (CL_SETTINGS["INVERT"..tree] == 1 and CritlineFilters_IsSpellInFilter(k, tree)) then
					hidmg = CL_DATABASE[tree][k][hittype]["Amount"]
					spell = k
				end
			end
		end
	end
	return hidmg, spell
end


function Critline_RebuildAllTooltips()
	Critline_Summary = {DMG = "", HEAL = "", PET = ""}	-- have to rebuild cached data when a spell is filtered to update tooltips
	Critline_GetSummaryRichText("DMG") 
	Critline_GetSummaryRichText("HEAL")
	Critline_GetSummaryRichText("PET")
	Critline_DoUpdate()
end


Critline_Summary = {DMG = "", HEAL = "", PET = ""}

function Critline_GetSummaryRichText(tree)
	-- This is for code optimization. We only build the rich summary when a new record is hit.
	-- Then a new record is recorded, the engine will blank out the Critline_Summary[tree]
	-- This causes the tree to be rebuilt when this function is called.
	-- tree is 'DMG', 'PET' or 'HEAL'
	
	tree = strupper(tree)
	if (Critline_Summary[tree] == "") then
		Critline_Summary[tree] = Critline_BuildSummaryRichText(tree)
	end
	return Critline_Summary[tree]
end


function Critline_BuildSummaryRichText(tree)
	-- tree is 'DMG', 'PET' or 'HEAL'
	
	tree = strupper(tree)
	local hicrit = Critline_GetHighest(tree, "CRIT")
	local hidmg = Critline_GetHighest(tree, "NORM")
	local rtfAttack = ""

	if not CL_DATABASE[tree] then 
		return SUBHEADER_TEXT_COLOR..CL_NORECORDS_TEXT..FONT_COLOR_CODE_CLOSE
	end
	
	local crit_name = ""
	local crit_level = ""
	local crit_amount = ""
	local crit_date = ""
	
	local norm_name = ""
	local norm_level = ""
	local norm_amount = ""
	local norm_date = ""

	local formattedEntry = ""
	local summaryFormat, spellName, normal, crit
	
	sort(CL_DATABASE[tree])

	-- $sn = Spell name

	-- $nn = Target's Name for the Normal Damage Record
	-- $nl = Target's Level for the Normal Damage Record
	-- $na = Amount of damage for the Normal spell.
	-- $nd = Date of Normal spell record.
	-- $nc = Number of times this spells has hit normal.

	-- $cn = Target's Name for the Crit Damage Record
	-- $cl = Target's Level for the Crit Damage Record
	-- $ca = Amount of damage for the Crit spell.
	-- $cd = Date of Crit spell record.
	-- $cc = Number of times this spells has hit Crit.
	if CL_SETTINGS["DETAILED"] == 0 then
		-- I would like these vales to be in a user editable on the settings menu (Advanced Settings)
		-- BUG:14
		summaryFormat = "$sn\t[$na/$ca]\n"
	else
		spellName = "$sn\n"
	end
	
	Critline_FuBar_Tooltip = {} -- always reset on each call
	
	for spell in pairs(CL_DATABASE[tree]) do
		if type(spell) == "number" then -- need to get rid of old 113 errors...they are stored as numbers and cause errors when compared with string names
			CL_DATABASE[tree][spell] = nil
		elseif (CL_SETTINGS["INVERT"..tree] == 0 and not CritlineFilters_IsSpellInFilter(spell, tree)) or (CL_SETTINGS["INVERT"..tree] == 1 and CritlineFilters_IsSpellInFilter(spell, tree)) then
			-- feyde...had to add brackets around damage for parsing in Fubar tooltip
			normal = "   "..CL_NORMAL_TEXT..": |cffffffff$nn|r ($nl)\t $na\n"
			crit = "   "..CL_CRIT_TEXT..": |cffffffff$cn|r ($cl)\t $ca\n"
			
			-- get norm_amount
			if CL_DATABASE[tree][spell]["NORM"] then
				if CL_DATABASE[tree][spell]["NORM"]["Amount"] == hidmg then
					norm_amount = GREEN_FONT_COLOR_CODE..CL_DATABASE[tree][spell]["NORM"]["Amount"]..FONT_COLOR_CODE_CLOSE
				else
					norm_amount = HIGHLIGHT_FONT_COLOR_CODE..CL_DATABASE[tree][spell]["NORM"]["Amount"]..FONT_COLOR_CODE_CLOSE
				end
				norm_name = CL_DATABASE[tree][spell]["NORM"]["Target"]
				norm_level = CL_DATABASE[tree][spell]["NORM"]["Level"]
				norm_date = CL_DATABASE[tree][spell]["NORM"]["Date"]
				norm_count = CL_DATABASE[tree][spell]["NORM"]["Count"]
			else
				norm_name = ""
				norm_level = ""
				norm_amount = HIGHLIGHT_FONT_COLOR_CODE.."0"..FONT_COLOR_CODE_CLOSE
				norm_date = ""
				norm_count = ""
				normal = ""
			end

			-- get crit_amount
			if CL_DATABASE[tree][spell]["CRIT"] then
				if CL_DATABASE[tree][spell]["CRIT"]["Amount"] == hicrit then
					crit_amount = GREEN_FONT_COLOR_CODE..CL_DATABASE[tree][spell]["CRIT"]["Amount"]..FONT_COLOR_CODE_CLOSE
				else
					crit_amount = HIGHLIGHT_FONT_COLOR_CODE..CL_DATABASE[tree][spell]["CRIT"]["Amount"]..FONT_COLOR_CODE_CLOSE
				end
				crit_name = CL_DATABASE[tree][spell]["CRIT"]["Target"]
				crit_level = CL_DATABASE[tree][spell]["CRIT"]["Level"]
				crit_date = CL_DATABASE[tree][spell]["CRIT"]["Date"]
				crit_count = CL_DATABASE[tree][spell]["CRIT"]["Count"]
			else
				crit_name = ""
				crit_level = ""
				crit_amount = HIGHLIGHT_FONT_COLOR_CODE.."0"..FONT_COLOR_CODE_CLOSE
				crit_date = ""
				crit_count = ""
				crit = ""
			end

			if CL_SETTINGS["DETAILED"] == 1 then
				formattedEntry = spellName..normal..crit
			else
				formattedEntry = summaryFormat
			end

			formattedEntry = gsub(formattedEntry, "$sn", spell)

			formattedEntry = gsub(formattedEntry, "$nn", norm_name)
			formattedEntry = gsub(formattedEntry, "$nl", norm_level)
			formattedEntry = gsub(formattedEntry, "$na", norm_amount)
			formattedEntry = gsub(formattedEntry, "$nd", norm_date)
			formattedEntry = gsub(formattedEntry, "$nc", norm_count)

			formattedEntry = gsub(formattedEntry, "$cn", crit_name)
			formattedEntry = gsub(formattedEntry, "$cl", crit_level)
			formattedEntry = gsub(formattedEntry, "$ca", crit_amount)
			formattedEntry = gsub(formattedEntry, "$cd", crit_date)
			formattedEntry = gsub(formattedEntry, "$cc", crit_count)

			rtfAttack = rtfAttack..formattedEntry
			
			-- below we will build a table for the FuBar tooltip
			-- it has been formatted the same as the summaryformat above.  if summaryformat changes, then this will need to change as well
			-- i am using "!!" as a left and right separator
			if Critline_FuBar_Active then -- feyde..only active when ANY Critline FuBar plugin is loaded
				if CL_SETTINGS["DETAILED"] == 0 then
					tinsert(Critline_FuBar_Tooltip, spell.."!!["..norm_amount.."/"..crit_amount.."]")
				else
					tinsert(Critline_FuBar_Tooltip, spell)
					tinsert(Critline_FuBar_Tooltip, "   "..CL_NORMAL_TEXT..": "..norm_name.." ("..norm_level..")!!"..norm_amount)
					tinsert(Critline_FuBar_Tooltip, "   "..CL_CRIT_TEXT..": "..crit_name.." ("..crit_level..")!!"..crit_amount)
				end
			end
		end
	end

	if rtfAttack == "" then
		tinsert(Critline_FuBar_Tooltip, SUBHEADER_TEXT_COLOR..CL_NORECORDS_TEXT..FONT_COLOR_CODE_CLOSE) --feyde...insert into  FuBar tooltip table
		return SUBHEADER_TEXT_COLOR..CL_NORECORDS_TEXT..FONT_COLOR_CODE_CLOSE
	end
	
	return strsub(rtfAttack, 1, (#rtfAttack - 1))
end


function Critline_OnEvent(self, event, ...)
	if event == "ADDON_LOADED" then
		if (...) == "Critline" then
			Critline_Initialize()
			CritlineSplashFrame:EnableMouse(0)
			CritlineSplashFrame:Clear()
			Critline_Loaded = true
		end
	elseif event == "PLAYER_LOGIN" then
		playerGUID = UnitGUID("player")
		Critline_Debug("playerGUID: "..playerGUID)
		
		inVehicle = UnitInVehicle("player")
		
		if CritlineFilters_HasSpecialAura() then
			hasSpecialAura = true
			Critline_Debug("Special aura found. Hits won't be recorded.")
		end
	elseif event == "UNIT_ENTERED_VEHICLE" then
		inVehicle = true
	elseif event == "UNIT_EXITED_VEHICLE" then
		inVehicle = false
	elseif event == "PLAYER_CONTROL_LOST" then
		if not Critline_Loaded then
			return
		end
		-- this covers fear, taxi, mind control, and others
		Critline_OutOfControl = true
	elseif event == "PLAYER_CONTROL_GAINED" then
		if not Critline_Loaded then
			return
		end
		Critline_OutOfControl = false
		Critline_MindControlled = false
	elseif event == "PLAYER_DEAD" then
		hasSpecialAura = CritlineFilters_HasSpecialAura()
	elseif event == "COMBAT_LOG_EVENT_UNFILTERED" then
		if not Critline_Loaded then
			return
		end
		
		local timestamp, eventType, sourceGUID, sourceName, sourceFlags, destGUID, destName, destFlags = ...
		local isPet
		-- we seem to get events with standard arguments equal to nil, so they need to be ignored
		if not (timestamp and eventType) then
			Critline_Debug("nil errors on start")
			return
		end

		-- if we don't have a destName (who we hit or healed) and we don't have a sourceName (us or our pets) then we leave
		if not (destName or sourceName) then
			Critline_Debug("nil source/dest")
			return
		end
		
		if destGUID == playerGUID then
			local spellID = select(9, ...)
			if (eventType == "SPELL_AURA_APPLIED" or eventType == "SPELL_AURA_REMOVED" or eventType == "SPELL_AURA_BROKEN" or eventType == "SPELL_AURA_BROkEN_SPELL") then
				for i, v in ipairs(Critline_SpecialAuras) do
					if v == spellID then
						if CritlineFilters_HasSpecialAura() then
							hasSpecialAura = true
							Critline_Debug("Special aura detected. Suppressing record tracking.")
						else
							hasSpecialAura = false
							Critline_Debug("No special aura detected. Resuming record tracking.")
						end
						return
					end
				end
			end
		end
		
		if hasSpecialAura then
			Critline_Debug("Special aura found. Return.")
			return
		end
		
		local myPetGUID = UnitGUID("pet")
		
		-- if sourceGUID is not us or our pet, we leave
		if sourceGUID ~= playerGUID then
			if myPetGUID then
				-- Critline_Debug("myPetGUID: "..myPetGUID)
				if sourceGUID ~= myPetGUID then
					-- Critline_Debug("sourceGUID("..sourceGUID..") not us("..playerGUID..") or our pet("..myPetGUID..")")
					return
				elseif inVehicle then
					Critline_Debug("In a vehicle, ignore.")
					return
				end
			else
				-- Critline_Debug("sourceGUID ("..sourceGUID..") ~= playerGUID ("..playerGUID.."). Return.")
				return
			end
		end
		
		if sourceGUID == myPetGUID then
			isPet = true
		end

		if (Critline_MindControlled and CL_SETTINGS["SUPPRESSMC"] == 1) then
			-- we are suppressing mind control events, time to leave
			Critline_Debug("Mind Control suppressed")
			return
		end
		
		if (eventType == "SPELL_CAST_START" or eventType == "SPELL_CAST_FAILED" or eventType == "SPELL_CAST_SUCCESS" or eventType == "SWING_DAMAGE") then
			if Critline_OutOfControl then
				-- if we are out of control and casting spells and/or attacking, we are mind controlled
				Critline_MindControlled = true
			end
		end
		if eventType == "SWING_DAMAGE" then
			-- can be non-Physical damage (e.g. Melee that hits as Shadow damage)
			local amount, _, school, _, _, _, critical = select(9, ...)
			if (amount and amount > 0) and not CritlineFilters_IsMobInFilter(destName, destGUID) then -- feyde..check for mob in filter
				Critline_HitHandle(sourceName, destName, "Melee", false, critical, amount, school, isPet, destGUID)
			end
		elseif (eventType == "RANGE_DAMAGE" or eventType == "SPELL_DAMAGE" or eventType == "SPELL_PERIODIC_DAMAGE" or eventType == "DAMAGE_SPLIT" or eventType == "DAMAGE_SHIELD") then
			local _, spellName, _, amount, _, school, _, _, _, critical = select(9, ...)
			if (amount and amount > 0) and not CritlineFilters_IsMobInFilter(destName, destGUID) then -- feyde..check for mob in filter
				Critline_HitHandle(sourceName, destName, spellName, false, critical, amount, school, isPet, destGUID)
			end
		-- healing events
		elseif (eventType == "SPELL_HEAL" or eventType == "SPELL_PERIODIC_HEAL") then
			local _, spellName, _, amount, _, critical = select(9, ...)
			if (amount and amount > 0) and not CritlineFilters_IsMobInFilter(destName, destGUID) then -- feyde..check for mob in filter
				Critline_HitHandle(sourceName, destName, spellName, true, critical, amount, 0, isPet, destGUID)
			end
		end
	
	end
end


function Critline_OnLoad(self)
	self:RegisterEvent("COMBAT_LOG_EVENT_UNFILTERED")
	self:RegisterEvent("ADDON_LOADED")
	self:RegisterEvent("PLAYER_LOGIN")
	self:RegisterEvent("PLAYER_CONTROL_LOST")
	self:RegisterEvent("PLAYER_CONTROL_GAINED")
	self:RegisterEvent("PLAYER_DEAD")
end


function Critline_Initialize()
	Critline_Debug("Initializing...")

	-- settings
	CL_SETTINGS = CL_SETTINGS or {}
	
	local defaultSettings = {
		-- SETTING = DefaultValue
		MINIMAPPOS = 0,
		SHOWMINIMAP = 1,
		SUPPRESSMC = 0,
		VERSION = CL_VERSION,
		LVLADJ = -1,
		SCALE = 1,
		SPLASHSCALE = 1,
		SPLASH = 1,
		RECORDPVE = 1,
		RECORDPVP = 0,
		PLAYSOUND = 1,
		SNAPSHOT = 0,
		DMG = 1,
		HEAL = 1,
		PET = 1,
		SHOWDMG = 1,
		SHOWHEAL = 1,
		SHOWPET = 1,
		INVERTDMG = 0,
		INVERTHEAL = 0,
		INVERTPET = 0,
		DONTFILTERMAGIC = 0,
		DETAILED = 0,
		FIRSTLOAD = 1,
		SPLASHTIMER = 2,
		SPELLCOLOR = {
			r = 1,
			g = 1,
			b = 0,
		},
		AMOUNTCOLOR = {
			r = 1,
			g = 1,
			b = 1,
		},
		SPLASHFRAME = {}
	}
	
	for setting, value in pairs(defaultSettings) do
		if not CL_SETTINGS[setting] then
			CL_SETTINGS[setting] = value
		elseif type(CL_SETTINGS[setting]) == "string" then
			CL_SETTINGS[setting] = tonumber(CL_SETTINGS[setting])
		end
	end
	
	if CL_SETTINGS["SPLASHFRAME"]["point"] then
		CritlineSplashFrame:ClearAllPoints()
		CritlineSplashFrame:SetPoint(CL_SETTINGS["SPLASHFRAME"]["point"], CL_SETTINGS["SPLASHFRAME"]["x"], CL_SETTINGS["SPLASHFRAME"]["y"])
	end
	
	-- database
	CL_DATABASE = CL_DATABASE or {}

	local trees = {"DMG", "HEAL", "PET"}

	for i, tree in ipairs(trees) do
		CL_DATABASE[tree] = CL_DATABASE[tree] or {}
	end
	
	-- filters
	if not CL_FILTERS then
		CL_FILTERS = {}
		Critline_Debug("filters created")
	end

	if not CritlineMobFilter then
		CritlineMobFilter = {}
		if CL_MOBFILTERS and CL_MOBFILTERS["mobs"] then
			CritlineMobFilter = CL_MOBFILTERS["mobs"]
		end
		Critline_Debug("Mob filter created.")
	end
	
	if not CL_FILTERS["DMG"] then
		for i, tree in ipairs(trees) do
			CL_FILTERS[tree] = {}
		end
		Critline_Debug("spells created")
	end

	CritlineSplashFrame:SetTimeVisible(CL_SETTINGS["SPLASHTIMER"])
	
	Critline_RebuildAllTooltips()
	Critline_Debug("Initialization Complete.")
end


local tooltip = CreateFrame("GameTooltip", "CritlineTooltip", nil, "GameTooltipTemplate")

function Critline_UnitLevel(destGUID)
	CritlineTooltip:SetOwner(UIParent, "ANCHOR_NONE")
	CritlineTooltip:SetHyperlink("unit:"..destGUID)
	local level = UnitLevel("player") -- by default, all mobs are fair game
	local isPlayer = false -- by default we are fighting a mob
	for i = 1, CritlineTooltip:NumLines() do
		local mytext = _G["CritlineTooltipTextLeft"..i]
		local text = mytext:GetText()
		if text then
			if strmatch(text, LEVEL) then -- our destGUID has the world Level in it.
				level = strmatch(text, "(%d+)")  -- find the level
				if level then  -- if we found the level, break from the for loop
					level = tonumber(level)
				else
					-- well, the word Level is in this tooltip, but we could not find the level
					-- either the destGUID is at least 10 levels higher than us, or we just couldn't find it.
					level = UnitLevel("player")  -- like I said, by default, all mobs are fair game
				end
			end
			if strmatch(text, PLAYER) then -- our destGUID has the word Player in it.
				-- we are fighting another Player, this is used for PvP flag
				isPlayer = true
			end
		end
	end	
	return level, isPlayer
end


function Critline_HitHandle(attacker, target, spell, isHeal, critical, amount, school, isPet, destGUID)
	local targetlvl, isPlayer = Critline_UnitLevel(destGUID)
	local tree

	if (not isPlayer and CL_SETTINGS["RECORDPVE"] == 0) or (isPlayer and CL_SETTINGS["RECORDPVP"] == 0) then
		Critline_Debug("Target == player/non player and not recording PvP/PvE; return.")
		return
	end

	local leveldiff = 0
	if targetlvl < UnitLevel("player") then
		leveldiff = (UnitLevel("player") - targetlvl)
	end
	
	Critline_Debug("Level difference: "..leveldiff)
	Critline_Debug("Level adjustment: "..CL_SETTINGS["LVLADJ"])
	Critline_Debug("Spell school: "..school)
	
	-- ignore level adjustment if magic damage
	if (CL_SETTINGS["LVLADJ"] >= 0) and (CL_SETTINGS["LVLADJ"] < leveldiff) and (school == 1 or CL_SETTINGS["DONTFILTERMAGIC"] == 0) then
		Critline_Debug("Level adjustment enabled, target level too low and damage school is filtered; return.")
		return
	end

	if not CL_DATABASE then
		Critline_Debug("CL_DATABASE should not be nil at this point!")
		Critline_Initialize()
	end

	if isPet then
		if not isHeal then
			tree = "PET"
		end
	elseif isHeal then
		tree = "HEAL"
	else
		tree = "DMG"
	end

	-- exit if not recording type dmg.
	if (not tree or CL_SETTINGS[tree] == 0) then
		return
	end
	
	local hitype = "NORM"
	if critical then
		hitype = "CRIT"
	end

	if not CL_DATABASE[tree] then
		Critline_Debug("Creating CL_DATABASE[\""..tree.."\"]...")
		CL_DATABASE[tree] = {}
	end

	if not CL_DATABASE[tree][spell] then
		Critline_Debug("Creating CL_DATABASE[\""..tree.."\"][\""..spell.."\"]...")
		CL_DATABASE[tree][spell] = {}
		CLASpellScrollBar_Update(CritlineAnnounceFrameSpellListScrollBar)	-- update announce..
		CLASpellScrollBar_Update(CritlineResetFrameSpellListScrollBar)	-- ..reset..
		CLFSpellScrollBar_Update()	-- ..and filter list
	end

	if not CL_DATABASE[tree][spell][hitype] then
		Critline_Debug("Creating CL_DATABASE[\""..tree.."\"][\""..spell.."\"][\""..hitype.."\"]...")
		CL_DATABASE[tree][spell][hitype] = {}
	end

	if not CL_DATABASE[tree][spell][hitype]["Amount"] then
		Critline_Debug("Creating CL_DATABASE[\""..tree.."\"][\""..spell.."\"][\""..hitype.."\"][\"Amount\"]...")
		CL_DATABASE[tree][spell][hitype]["Amount"] = 0
	end

	if not CL_DATABASE[tree][spell][hitype]["Count"] then
		Critline_Debug("Creating CL_DATABASE[\""..tree.."\"][\""..spell.."\"][\""..hitype.."\"][\"Count\"]...")
		CL_DATABASE[tree][spell][hitype]["Count"] = 0
	end

	CL_DATABASE[tree][spell][hitype]["Count"] = CL_DATABASE[tree][spell][hitype]["Count"] + 1

	if (not CL_DATABASE[tree][spell][hitype]["Amount"] or CL_DATABASE[tree][spell][hitype]["Amount"] < amount) then
		CL_DATABASE[tree][spell][hitype]["Amount"] = amount
		CL_DATABASE[tree][spell][hitype]["Target"] = target
		CL_DATABASE[tree][spell][hitype]["Level"] = targetlvl
		CL_DATABASE[tree][spell][hitype]["Date"] = date()
		Critline_Summary[tree] = ""
		if (CritlineFilters_IsSpellInFilter(spell, tree) and CL_SETTINGS["INVERT"..tree] == 1) or (not CritlineFilters_IsSpellInFilter(spell, tree) and CL_SETTINGS["INVERT"..tree] == 0) then
			Critline_DisplayNewRecord(spell, amount, critical)
		end
	end
end


function Critline_DisplayNewRecord(spell, amount, critical)
	splash_msg = CL_NEW_RECORD_MSG
	if CL_SETTINGS["SPLASH"] == 1 then
		CritlineSplashFrame:Clear()
		CritlineSplashFrame:AddMessage(amount, CL_SETTINGS["AMOUNTCOLOR"]["r"], CL_SETTINGS["AMOUNTCOLOR"]["g"], CL_SETTINGS["AMOUNTCOLOR"]["b"], 1)
		CritlineSplashFrame:AddMessage(format(splash_msg, spell), CL_SETTINGS["SPELLCOLOR"]["r"], CL_SETTINGS["SPELLCOLOR"]["g"], CL_SETTINGS["SPELLCOLOR"]["b"], 1)
		if critical then
			CritlineSplashFrame:AddMessage(strupper(strmatch(TEXT_MODE_A_STRING_RESULT_CRITICAL, "(%a+)")).."!", 1, 0, 0, 1)
		end
	end
	if CL_SETTINGS["PLAYSOUND"] == 1 then 
		PlaySound("LEVELUP", 1, 1, 0, 1, 3) 
	end
	if CL_SETTINGS["SNAPSHOT"] == 1 then 
		-- To avoid protection issues, you could substitute SetAlpha(0) and SetAlpha(100) for Hide() and Show(), respectively.
		-- if UIParent:IsShown() 
		-- UIParent:Hide() --SetAlpha(0)
		TakeScreenshot() 
		-- UIParent:Show() --SetAlpha(100)
	end
	Critline_RebuildAllTooltips() --feyde...since Tooltips are now cached, we have to reset them to update the new record data.
end


CLOU_Handles = {}

function Critline_DoUpdate()
	for k, v in pairs(CLOU_Handles) do
		v()
	end
end


function Critline_OnUpdateRegister(newhandler)
	tinsert(CLOU_Handles, newhandler)
end


--[[ misc help functions ]]
function Critline_Color(color, msg)
	if msg then
		return color..msg..FONT_COLOR_CODE_CLOSE
	end
end


function Critline_Debug(message)
	if (CL_DEBUG and DEFAULT_CHAT_FRAME) then
		DEFAULT_CHAT_FRAME:AddMessage("CL_DEBUG: "..message)
	end
end


function Critline_Message(msg)
	if not msg then
		msg = "Critline_Message passed a nil value."
	end
	if DEFAULT_CHAT_FRAME then
		DEFAULT_CHAT_FRAME:AddMessage(msg)
	end
end


-- xml function calls
function CritlineSplashFrame_OnLoad(self)
	self.update = 1
	self.locked = true
	self:RegisterForDrag("LeftButton")
end


function CritlineSplashFrame_OnUpdate(self)
	if not Critline_Loaded then
		return
	end

	if not self.locked then
		CritlineSplashFrame:AddMessage(CL_SPLASHFRAME_UNLOCKED_TEXT, CL_SETTINGS["AMOUNTCOLOR"]["r"], CL_SETTINGS["AMOUNTCOLOR"]["g"], CL_SETTINGS["AMOUNTCOLOR"]["b"], 1)
		CritlineSplashFrame:AddMessage(CL_SPLASHFRAME_DRAGTOMOVE_TEXT, CL_SETTINGS["SPELLCOLOR"]["r"], CL_SETTINGS["SPELLCOLOR"]["g"], CL_SETTINGS["SPELLCOLOR"]["b"], 1)
	end
end


function CritlineSplashFrame_OnDragStart(self)
	if not self.locked then
		self:StartMoving()
	end
end


function CritlineSplashFrame_OnDragStop(self)
	self:StopMovingOrSizing()
	local point, relativeTo, relativePoint, xOfs, yOfs = CritlineSplashFrame:GetPoint()
	CL_SETTINGS["SPLASHFRAME"]["point"] = point
	CL_SETTINGS["SPLASHFRAME"]["x"] = xOfs
	CL_SETTINGS["SPLASHFRAME"]["y"] = yOfs
end