--[[
Name: FishLib-1.0
Author(s): Sutorix <sutorix@hotmail.com>
Description: A library with common routines used by FishingBuddy and addons.
--]]

local MAJOR_VERSION = "LibFishing-1.0"
local MINOR_VERSION = 1

if not LibStub then error(MAJOR_VERSION .. " requires LibStub") end

local FishLib, oldLib = LibStub:NewLibrary(MAJOR_VERSION, MINOR_VERSION)
if not FishLib then
	return
end

local WOW = {};
function FishLib:WOWVersion()
   return WOW.major, WOW.minor, WOW.dot;
end

if ( GetBuildInfo ) then
   local v, b, d = GetBuildInfo();
   WOW.build = b;
   WOW.date = d;
   local s,e,maj,min,dot = string.find(v, "(%d+).(%d+).(%d+)");
   WOW.major = tonumber(maj);
   WOW.minor = tonumber(min);
   WOW.dot = tonumber(dot);
else
   WOW.major = 1;
   WOW.minor = 9;
   WOW.dot = 0;
end

local Crayon = LibStub("LibCrayon-3.0");

local function FixupThis(target, tag, what)
   if ( type(what) == "table" ) then
      for idx,str in pairs(what) do
         what[idx] = FixupThis(target, tag, str);
      end
      return what;
   elseif ( type(what) == "string" ) then
      local pattern = "#([A-Z0-9_]+)#";
      local s,e,w = string.find(what, pattern);
      while ( w ) do
         if ( type(target[w]) == "string" ) then
            local s1 = strsub(what, 1, s-1);
            local s2 = strsub(what, e+1);
            what = s1..target[w]..s2;
            s,e,w = string.find(what, pattern);
         elseif ( Crayon and Crayon["COLOR_HEX_"..w] ) then
            local s1 = strsub(what, 1, s-1);
            local s2 = strsub(what, e+1);
            what = s1.."ff"..Crayon["COLOR_HEX_"..w]..s2;
            s,e,w = string.find(what, pattern);
         else
            -- stop if we can't find something to replace it with
            w = nil;
         end
      end
      return what;
   else
      FishingBuddy.Debug("tag "..tag.." type "..type(what));
      FishingBuddy.Dump(what);
   end
end

function FishLib:FixupEntry(constants, tag)
   FixupThis(constants, tag, constants[tag]);
end

local function FixupStrings(source, target)
   local translation = source["enUS"];
   for tag,_ in pairs(translation) do
      target[tag] = FixupThis(target, tag, target[tag]);
   end
end

local function FixupBindings(source, target)
   local translation = source["enUS"];
   for tag,str in pairs(translation) do      
      if ( string.find(tag, "^BINDING") ) then
         setglobal(tag, target[tag]);
         target[tag] = nil;
      end
   end
end

local missing = {};
local function LoadTranslation(source, lang, target, record)
   local translation = source[lang];
   if ( translation ) then
      for tag,value in pairs(translation) do
         if ( not target[tag] ) then
            target[tag] = value;
            if ( record ) then
               missing[tag] = 1;
            end
         end
      end
   end
end

function FishLib:Translate(addon, source, target, record)
   local locale = GetLocale();
   --locale = "deDE";
   target.VERSION = GetAddOnMetadata(addon, "Version");
   LoadTranslation(source, locale, target);
   if ( locale ~= "enUS" ) then
      LoadTranslation(source, "enUS", target, record);
   end
   FixupStrings(source, target);
   FixupBindings(source, target);
end

local itempattern = "|c(%x+)|Hitem:(%d+)(:%d+):%d+:%d+:%d+:%d+:[-]?%d+:[-]?%d+:[-]?%d+|h%[(.*)%]|h|r";
function FishLib:SplitLink(link)
   if ( link ) then
      local _,_, color, id, item, name = string.find(link, itempattern);
      return color, id..item, name;
   end
end

function FishLib:SplitFishLink(link)
   if ( link ) then
      local _,_, color, id, item, name = string.find(link, itempattern);
      return color, tonumber(id), name;
   end
end

function FishLib:GetItemInfo(link)
   local maj,min,dot = FishLib:WOWVersion();
-- name, link, rarity, itemlevel, minlevel, itemtype
-- subtype, stackcount, equiploc, texture
   local nm,li,ra,il,ml,it,st,sc,el,tx;
   if ( maj > 1 ) then
      nm,li,ra,il,ml,it,st,sc,el,tx = GetItemInfo(link);
   else
      nm,li,ra,ml,it,st,sc,el,tx = GetItemInfo(link);
   end
   return nm,li,ra,ml,it,st,sc,el,tx,il;
end

function FishLib:IsLinkableItem(item)
   local link = "item:"..item;
   local n,l,_,_,_,_,_,_ = FishLib:GetItemInfo(link);
   return ( n and l );
end

function FishLib:ChatLink(item, name, color)
   if( item and name and ChatFrameEditBox:IsVisible() ) then
      if ( not color ) then
         color = Crayon.COLOR_HEX_WHITE;
      elseif ( Crayon["COLOR_HEX_"..color] ) then
         color = Crayon["COLOR_HEX_"..color];
      end
      if ( string.len(color) == 6) then
         color = "ff"..color;
      end
      local link = "|c"..color.."|Hitem:"..item.."|h["..name.."]|h|r";
      ChatFrameEditBox:Insert(link);
   end
end

function FishLib:IsFishingPole()
   -- Get the main hand item texture
   local slot = GetInventorySlotInfo("MainHandSlot");
   local itemTexture = GetInventoryItemTexture("player", slot);
   -- If there is infact an item in the main hand, and it's texture
   -- that matches the fishing pole texture, then we have a fishing pole
   if ( itemTexture and string.find(itemTexture, "INV_Fishingpole") ) then
      local link = GetInventoryItemLink("player", slot);
      local _, id, _ = FishLib:SplitFishLink(link);
      -- Make sure it's not "Nat Pagle's Fish Terminator"
      if ( id ~= 19944) then
         return true;
      end
   end
   return false;
end

function FishLib:IsWorn(itemid)
   for slot=1,19 do
      local link = GetInventoryItemLink("player", slot);
      if ( link ) then
         local _, id, _ = FishLib:SplitFishLink(link);
         if ( itemid == id ) then
            return true;
         end
      end
   end
   -- return nil
end

local function SplitColor(color)
   if ( color ) then
      if ( type(color) == "table" ) then
         for i,c in pairs(color) do
            color[i] = SplitColor(c);
         end
      elseif ( type(color) == "string" ) then
         local a = tonumber(string.sub(color,1,2),16);
         local r = tonumber(string.sub(color,3,4),16);
         local g = tonumber(string.sub(color,5,6),16);
         local b = tonumber(string.sub(color,7,8),16);
         color = { a = a, r = r, g = g, b = b };
      end
   end
   return color;
end

-- utility functions
local function AddTooltipLine(l)
   if ( type(l) == "table" ) then
      -- either { t, c } or {{t1, c1}, {t2, c2}}
      if ( type(l[1]) == "table" ) then
         local c1 = SplitColor(l[1][2]) or {};
         local c2 = SplitColor(l[2][2]) or {};
         GameTooltip:AddDoubleLine(l[1][1], l[2][1],
                                   c1.r, c1.g, c1.b,
                                   c2.r, c2.g, c2.b);
      else
         local c = SplitColor(l[2]) or {};
         GameTooltip:AddLine(l[1], c.r, c.g, c.b, 1);
      end
   else
      GameTooltip:AddLine(l,nil,nil,nil,1);
   end
end

function FishLib:AddTooltip(text)
   local c = color or {{}, {}};
   if ( text ) then
      if ( type(text) == "table" ) then
         for _,l in pairs(text) do
            AddTooltipLine(l);
         end
      else
         -- AddTooltipLine(text, color);
         GameTooltip:AddLine(text,nil,nil,nil,1);
      end
   end
end

